import pandas as pd
import json
import math
import numpy as np
import re
import sys
import os
import traceback
from pathlib import Path
from datetime import datetime
from multiprocessing import cpu_count

# Mēģinām importēt bibliotēkas vizualizācijai/paralelizācijai
try:
    from tqdm import tqdm
    tqdm.pandas(desc="Datu apstrāde")
    from pandarallel import pandarallel
except ImportError:
    print("KĻŪDA: Trūkst nepieciešamās bibliotēkas. Lūdzu, instalējiet tās:")
    print("pip3 install tqdm pandarallel numpy pandas")
    sys.exit(1)

# --- 1. KONFIGURĀCIJA UN CEĻI ---

# Skripta atrašanās vieta
SCRIPT_DIR = Path(__file__).parent.resolve()

# Ievades direktorijas
CSV_DIR = SCRIPT_DIR / 'csv'         
KODS_DIR = SCRIPT_DIR / 'kods'       

# Izvades direktorijas
OUTPUT_DIR = KODS_DIR / 'pozicija'          # Šeit būs nace_list.json
JSON_SUBDIR = OUTPUT_DIR / 'json'           # Šeit būs pārējie faili (apakšdirektorija)

# Failu ceļi
NACE_FILE = KODS_DIR / 'NACE.csv'
REGISTER_FILE = CSV_DIR / 'register.csv'
TAX_DATA_FILE = CSV_DIR / 'pdb_nm_komersantu_samaksato_nodoklu_kopsumas_odata.csv'
FINANCIAL_STATEMENTS_FILE = CSV_DIR / 'financial_statements.csv'
INCOME_STATEMENTS_FILE = CSV_DIR / 'income_statements.csv'
BALANCE_SHEETS_FILE = CSV_DIR / 'balance_sheets.csv'

# Izvades fails (saraksts paliek galvenajā mapē)
NACE_LIST_JSON = OUTPUT_DIR / 'nace_list.json'

# --- 2. IMPORTI NO CITIEM MODUĻIEM ---
try:
    PROJECT_ROOT = SCRIPT_DIR.parent
    sys.path.insert(0, str(PROJECT_ROOT))
    from core.page_builder import _calculate_net_salary
except ImportError:
    def _calculate_net_salary(gross, year):
        return {'net_salary': gross * 0.7} 

# --- 3. DATU IELĀDES UN APSTRĀDES FUNKCIJA ---

def load_and_process_data():
    print(f"--- Sāk datu apstrādi ---")
    print(f"CSV direktorija: {CSV_DIR}")
    print(f"NACE fails: {NACE_FILE}")

    # 1. Pārbaudes
    if not CSV_DIR.exists():
        print(f"KĻŪDA: Nav atrasta 'csv' direktorija: {CSV_DIR}")
        sys.exit(1)
    if not KODS_DIR.exists():
        print(f"KĻŪDA: Nav atrasta 'kods' direktorija: {KODS_DIR}")
        sys.exit(1)
    
    # Izveido izvades mapes
    OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
    # Apakšdirektoriju izveidosim vēlāk ģenerēšanas solī, vai arī šeit:
    JSON_SUBDIR.mkdir(parents=True, exist_ok=True)

    # 2. Ielādē NACE nosaukumus
    print("Ielādē NACE klasifikatoru...")
    try:
        nace_df = pd.read_csv(
            NACE_FILE, 
            dtype={'Kods': str, 'Nosaukums': str}, 
            usecols=['Kods', 'Nosaukums']
        )
        nace_df['Kods'] = nace_df['Kods'].str.strip()
        nace_df['Nosaukums'] = nace_df['Nosaukums'].str.strip()
        nace_map = pd.Series(nace_df.Nosaukums.values, index=nace_df.Kods).to_dict()
    except Exception as e:
        print(f"KĻŪDA ielādējot NACE.csv: {e}")
        sys.exit(1)

    # 3. Ielādē Reģistra datus
    print("Ielādē un filtrē Uzņēmumu reģistra datus...")
    try:
        register_df = pd.read_csv(
            REGISTER_FILE, delimiter=';',
            usecols=['regcode', 'name', 'closed', 'terminated', 'type', 'name_in_quotes', 'registered'],
            dtype=str
        )
        
        register_df['regcode'] = register_df['regcode'].str.strip()
        register_df['closed'] = register_df['closed'].fillna('').str.strip()
        register_df['terminated'] = register_df['terminated'].fillna('').str.strip()
        
        active_mask = (register_df['closed'] != 'L') & (register_df['closed'] != 'R') & \
                      ((register_df['terminated'] == '') | (register_df['terminated'] == '0000-00-00'))
        active_register_df = register_df[active_mask].copy()
        
        def format_name(row):
            type_val = str(row.get('type', '')).strip()
            name_in_quotes = str(row.get('name_in_quotes', '')).strip()
            if type_val and name_in_quotes: return f"{type_val} \"{name_in_quotes}\""
            elif name_in_quotes: return f"\"{name_in_quotes}\""
            return str(row.get('name', '')).strip()

        tqdm.pandas(desc="Formatē nosaukumus")
        active_register_df['name'] = active_register_df.progress_apply(format_name, axis=1)
        
        main_df = active_register_df[['regcode', 'name', 'registered']].copy()
        
    except Exception as e:
        print(f"KĻŪDA ielādējot register.csv: {e}")
        sys.exit(1)

    # 4. Ielādē un pievieno NACE kodus no VID datiem (Tax data)
    print("Ielādē VID nodokļu datus (NACE kodi un darbinieki)...")
    try:
        tax_df = pd.read_csv(
            TAX_DATA_FILE, delimiter=',',
            usecols=['Registracijas_kods', 'Pamatdarbibas_NACE_kods', 'Taksacijas_gads', 'Videjais_nodarbinato_personu_skaits_cilv'],
            dtype={'Registracijas_kods': str, 'Pamatdarbibas_NACE_kods': str, 'Taksacijas_gads': str}
        )
        tax_df.rename(columns={'Registracijas_kods': 'regcode'}, inplace=True)
        tax_df['regcode'] = tax_df['regcode'].str.strip()
        
        # --- JAUNĀ LOĢIKA '?' APSTRĀDEI ---
        
        # Notīrām atstarpes
        tax_df['Pamatdarbibas_NACE_kods'] = tax_df['Pamatdarbibas_NACE_kods'].str.strip()
        
        # Aizstājam '?' ar NaN
        tax_df.loc[tax_df['Pamatdarbibas_NACE_kods'] == '?', 'Pamatdarbibas_NACE_kods'] = np.nan
        
        # Pārveidojam gadu par skaitli
        tax_df['year_int'] = pd.to_numeric(tax_df['Taksacijas_gads'], errors='coerce').fillna(0).astype(int)
        
        # Kārtojam: Vispirms pēc RegCode, tad pēc Gada (dilstoši - jaunākais augšā)
        tax_df.sort_values(by=['regcode', 'year_int'], ascending=[True, False], inplace=True)
        
        # Aizpildām tukšos NACE kodus no vecākiem gadiem (Backward Fill)
        # Tā kā dati sakārtoti [2023, 2022], bfill ņems no 2022 un ieliks 2023 (ja 2023 ir NaN)
        tax_df['Pamatdarbibas_NACE_kods'] = tax_df.groupby('regcode')['Pamatdarbibas_NACE_kods'].bfill()
        
        # --- BEIGAS JAUNAJAI LOĢIKAI ---

        # Tagad droši ņemam pirmo (jaunāko) ierakstu katram uzņēmumam
        latest_tax_data = tax_df.drop_duplicates(subset=['regcode'], keep='first').copy()
        
        # Sagatavojam NACE kodus
        latest_tax_data['nace_code_np'] = latest_tax_data['Pamatdarbibas_NACE_kods'].str.replace('.', '', regex=False)
        
        # Pievieno pamatdatiem
        main_df = pd.merge(main_df, latest_tax_data[['regcode', 'nace_code_np', 'Videjais_nodarbinato_personu_skaits_cilv']], on='regcode', how='left')
        main_df['nace_code_np'] = main_df['nace_code_np'].fillna('UNDEFINED')
        
    except Exception as e:
        print(f"KĻŪDA ielādējot nodokļu datus: {e}")
        traceback.print_exc()

    # 5. Ielādē Finanšu datus
    print("Ielādē un apstrādā Gada pārskatus (Finanšu dati)...")
    try:
        fs_df = pd.read_csv(
            FINANCIAL_STATEMENTS_FILE, delimiter=';',
            usecols=['legal_entity_registration_number', 'year', 'id', 'employees', 'rounded_to_nearest', 'source_type'],
            dtype=str
        )
        fs_df.rename(columns={'id': 'statement_id', 'legal_entity_registration_number': 'regcode'}, inplace=True)
        fs_df['regcode'] = fs_df['regcode'].str.strip()
        
        def set_priority(source_type):
            st = str(source_type).upper().strip()
            if st == 'UGP': return 1
            elif st == 'UKGP': return 2
            else: return 3
        fs_df['priority'] = fs_df['source_type'].apply(set_priority)
        
        fs_df.sort_values(by=['year', 'priority'], ascending=[False, True], inplace=True)
        latest_fs = fs_df.drop_duplicates(subset=['regcode'], keep='first').copy()
        
        income_df = pd.read_csv(INCOME_STATEMENTS_FILE, delimiter=';', dtype=str).set_index('statement_id')
        latest_fs = latest_fs.join(income_df[['net_turnover', 'net_income']], on='statement_id')
        
        def get_factor(rounding_str):
            rs = str(rounding_str).upper().strip()
            if rs == 'THOUSANDS': return 1000
            if rs == 'MILLIONS': return 1000000
            return 1
        latest_fs['factor'] = latest_fs['rounded_to_nearest'].apply(get_factor)
        
        for col in ['net_turnover', 'net_income']:
            latest_fs[col] = pd.to_numeric(latest_fs[col], errors='coerce') * latest_fs['factor']
            
        latest_fs['employees_fs'] = pd.to_numeric(latest_fs['employees'], errors='coerce').fillna(0).astype(int)
        
        main_df = pd.merge(main_df, latest_fs[['regcode', 'net_turnover', 'net_income', 'employees_fs']], on='regcode', how='left')
        
        # 6. Finalizēšana
        print("Finalizē aprēķinus...")
        
        main_df['employees_vid'] = pd.to_numeric(main_df['Videjais_nodarbinato_personu_skaits_cilv'], errors='coerce').fillna(0)
        main_df['employees'] = np.where(main_df['employees_fs'] > 0, main_df['employees_fs'], main_df['employees_vid'])
        main_df['employees'] = main_df['employees'].fillna(0).astype(int)
        
        main_df.rename(columns={
            'net_income': 'profit',
            'net_turnover': 'turnover'
        }, inplace=True)
        
        main_df['profit'] = main_df['profit'].fillna(0)
        main_df['turnover'] = main_df['turnover'].fillna(0)
        
        cols_to_keep = ['regcode', 'name', 'nace_code_np', 'profit', 'turnover', 'employees']
        final_df = main_df[cols_to_keep].copy()
        
        return final_df, nace_map

    except Exception as e:
        print(f"KĻŪDA finanšu datu apstrādē: {e}")
        traceback.print_exc()
        sys.exit(1)

# --- 5. JSON ĢENERĒŠANA ---

def generate_json_files(df, nace_map):
    print(f"\n--- Sāk JSON failu ģenerēšanu ---")
    print(f"Galvenais fails: {NACE_LIST_JSON}")
    print(f"Datu failu mape: {JSON_SUBDIR}")
    
    # Pārliecināmies, ka apakšdirektorija eksistē
    JSON_SUBDIR.mkdir(parents=True, exist_ok=True)

    # A. NACE LIST (Saglabājam 'pozicija' mapē)
    used_nace_codes = df['nace_code_np'].unique()
    nace_list_data = []
    sorted_codes = sorted([str(c) for c in used_nace_codes], key=lambda x: (x == 'UNDEFINED', x))
    
    for code in sorted_codes:
        name = "Bez NACE koda"
        if code != 'UNDEFINED':
            if code in nace_map:
                name = nace_map[code]
            else:
                dotted_code = f"{code[:2]}.{code[2:]}" if len(code) >= 4 else code
                name = nace_map.get(dotted_code, nace_map.get(code, "Nezināma nozare"))
        
        nace_list_data.append([code, name])

    try:
        with open(NACE_LIST_JSON, 'w', encoding='utf-8') as f:
            json.dump(nace_list_data, f, ensure_ascii=False, indent=2)
        print(f"Izveidots '{NACE_LIST_JSON.name}' ar {len(nace_list_data)} ierakstiem.")
    except IOError as e:
        print(f"KĻŪDA rakstot nace_list.json: {e}")

    # B. DATU FAILI (Saglabājam 'pozicija/JSON' apakšmapē)
    base_meta_data = {
        "generated": datetime.now().strftime('%Y-%m-%d'),
        "unit": "EUR",
        "description": "Uzņēmumu dati",
        "dimensions": ["name", "regcode", "profit", "turnover", "employees"]
    }
    
    file_count = 0
    grouped = df.groupby('nace_code_np')
    
    for nace_code, group in tqdm(grouped, desc="Ģenerē JSON failus"):
        sorted_group = group.sort_values(by='profit', ascending=False)
        companies_list = sorted_group[['name', 'regcode', 'profit', 'turnover', 'employees']].values.tolist()
        
        output_data = {
            "meta": base_meta_data.copy(),
            "companies": companies_list
        }
        output_data["meta"]["nace_code"] = nace_code
        
        # IZMAIŅA: Saglabājam apakšdirektorijā
        file_path = JSON_SUBDIR / f"{nace_code}.json"
        
        try:
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(output_data, f, ensure_ascii=False)
            file_count += 1
        except IOError:
            print(f"Kļūda rakstot {file_path}")

    print(f"\nPabeigts! Uzģenerēti {file_count} faili direktorijā '{JSON_SUBDIR}'.")

if __name__ == "__main__":
    companies_df, nace_mapping = load_and_process_data()
    generate_json_files(companies_df, nace_mapping)